﻿using System;
using System.IO;
using System.Threading.Tasks;
using Windows.Devices.PointOfService;
using Windows.Graphics.Imaging;
using Windows.Storage.Streams;
using Windows.Storage;

namespace AppUWP1
{
	public class HelperPosPrinterImageResize
	{
		internal static async Task<IRandomAccessStream> ConvertImageBytesToStream(byte[] imageBytes)
		{
			if (imageBytes == null)
				return null;
			InMemoryRandomAccessStream randomAccessStream = new InMemoryRandomAccessStream();

			Stream stream = randomAccessStream.AsStream();

			await stream.WriteAsync(imageBytes, 0, imageBytes.Length);
			await stream.FlushAsync();

			return randomAccessStream;
		}

		public static async Task<SoftwareBitmap> ResizeBitmap(IRandomAccessStream stream, BitmapFrame bitmapFrame, uint newWidth, uint newHeight)
		{
			var decoder = await BitmapDecoder.CreateAsync(stream);
			BitmapTransform bitmapTransform = new BitmapTransform();

			bitmapTransform.ScaledWidth = newWidth;
			bitmapTransform.ScaledHeight = newHeight;

			return await decoder.GetSoftwareBitmapAsync(
								bitmapFrame.BitmapPixelFormat,
								bitmapFrame.BitmapAlphaMode,
								bitmapTransform,
								ExifOrientationMode.IgnoreExifOrientation,
								ColorManagementMode.DoNotColorManage);
		}

		public static async Task<IRandomAccessStream> ConvertSoftwareBitmapToStream(SoftwareBitmap softwareBitmap)
		{
			try
			{
				InMemoryRandomAccessStream randomAccessStream = new InMemoryRandomAccessStream();
				BitmapEncoder encoder = await BitmapEncoder.CreateAsync(BitmapEncoder.PngEncoderId, randomAccessStream);
				encoder.SetSoftwareBitmap(softwareBitmap);
				await encoder.FlushAsync();

				randomAccessStream.Seek(0);
				return randomAccessStream;
			}
			catch (Exception ex)
			{
			}
			return null;
		}

		public static async Task<IRandomAccessStream> ConvertFileToStream(StorageFolder storageFolder, string filename)
		{
			try
			{
				StorageFile file = await storageFolder.GetFileAsync(filename);
				return await file.OpenAsync(FileAccessMode.Read);
			}
			catch (Exception ex)
			{
			}
			return null;
		}

		public static async Task<BitmapFrame> ConvertStreamToBitmap(IRandomAccessStream stream)
		{
			try
			{
				BitmapDecoder bitmapDecoder = await BitmapDecoder.CreateAsync(stream);
				return await bitmapDecoder.GetFrameAsync(0);
			}
			catch (Exception ex)
			{
			}
			return null;
		}

		public static async Task PrintImageWithResize(ReceiptPrintJob job, IRandomAccessStream imageStream, PosPrinterAlignment align, uint maxWidth)
		{
			if (job == null || imageStream == null || maxWidth == 0)
				return;

			var bitmapFrame = await ConvertStreamToBitmap(imageStream);

			maxWidth = maxWidth - 1;
			float rate = (float)maxWidth / (float)bitmapFrame.OrientedPixelWidth;
			uint newHeight = (uint)(bitmapFrame.OrientedPixelHeight * rate);

			var softwareBitmap = await ResizeBitmap(imageStream, bitmapFrame, maxWidth, newHeight);
			var streamResized = await ConvertSoftwareBitmapToStream(softwareBitmap);
			var bitmapFrameResized = await ConvertStreamToBitmap(streamResized);

			try
			{
				job.PrintBitmap(bitmapFrameResized, align);
			}
			catch (Exception ex)
			{
			}
		}
	}
}
